﻿/*! \file adc.c

	\brief	Routines for handling the ADC
	
	\copyright Copyright (C) 2009 Robert Loos	<http://www.loosweb.de>

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>
#include <math.h>
#include "adc.h"
#include "display.h"
#include "led.h"

static uint8_t	channel;	// The Channel which is actually in Conversion
static uint8_t	muxChannel;	// The Channel for the next Conversion
volatile uint8_t	adcFlags;

// ADC Raw Values
volatile ADCData_t	ADCData;	///< ADC-Values for the last measurement
volatile ADCData_t	ADCDataAvg;	///< Averaged ADC-Values
volatile ADCData_t	ADCDataMin;	///< Minimum ADC-Values during last average Period
volatile ADCData_t	ADCDataMinTmp;	///< Minimum ADC-Values in the current Period
volatile ADCData_t	ADCDataMax;	///< Maximum ADC-Values during last average Period
volatile ADCData_t	ADCDataMaxTmp;	///< Maximum ADC-Values in the current Period


/*!	\brief	Initializes the ADC

The ADC is set to free running mode at 115.2kHz clock.
This gives one result every 113us or one complete set
every 564us.
*/
void	InitADC(void)
{
	ADMUX=0xc0;	// Select channel 0, internal 2.56V reference
	channel=0;
	muxChannel=1;
	// ADCSRA: ADEN ADSC ADFR ADIF ADIE ADPS2 ADPS1 ADPS0
	ADCSRA=0b11001101;	// ADC enable, free running, Interrupt enable, CLK/32 (115.2kHz)
}

/*!	\brief	ADC Conversion Complete Interrupt

Calculates the Value of the current channel and
switches MUX and REF to the next channel.<br>
The flag ADCFLAG_NEWRESULTS (all channels have been updated) will
be set every 564us, ADCFLAG_NEWAVERAGES will be available every
36.1ms.<br>
Optimization for Size (-Os): This routine takes about 37us worst case.
*/
ISR(ADC_vect)
{
	static ADCData_t	ADCAvgTmp;
	static uint8_t	ADCAvgCnt;	// Counts the number of values added to ADCAvgTemp[]
	uint16_t	result;
	
	LEDRedOn();
	result=ADC;
/*	channel=ADMUX&0x1f;
	if (channel>4)	// Should not happen!!!
	{
		channel=4;	// Keep in limits
	}*/
	ADCData.array[channel]=result;
	ADCAvgTmp.array[channel]+=result;
	if (result<ADCDataMinTmp.array[channel])
	{
		ADCDataMinTmp.array[channel]=result;
	}
	if (result>ADCDataMaxTmp.array[channel])
	{
		ADCDataMaxTmp.array[channel]=result;
	}
	if (ADCAvgCnt==63)
	{
		ADCDataAvg.array[channel]=ADCAvgTmp.array[channel]/64;
		ADCAvgTmp.array[channel]=0;
	}			

	channel=muxChannel;
	muxChannel++;
	if (muxChannel>4)
	{
		muxChannel=0;
		adcFlags|=ADCFLAG_NEWRESULTS;
		ADCAvgCnt++;
		if (ADCAvgCnt==64)
		{
			ADCAvgCnt=0;
			adcFlags|=ADCFLAG_NEWAVERAGES;
		}			
	}
	ADMUX=0xC0|muxChannel;
	asm("NOP");
	ADCSRA|=0b01000000;
	LEDRedOff();
}